import { expect } from 'chai';
import { describe } from 'mocha';
import { deployContractsForTests, getSigners } from './helpers/deploy-helper';
import { OfferingToken } from '../../types';
import { BigNumber } from 'ethers';

describe('21 - Offering Token Contract Test', function () {
  let offeringToken: OfferingToken;
  let owner, addr1;

  async function addNMultipleAssets(assetCount: number) {
    const tokenUriBase = 'http://tokenuri'; // Base URI for tokens
    const assets = [];

    // [owner] = await ethers.getSigners();

    for (let i = 1; i <= assetCount; i++) {
      const assetId = `aabc${i}`;
      const oid = `oabc${i}`;
      const tokenUri = `${tokenUriBase}${i}.jpg`;

      await offeringToken.addAsset(
        assetId,
        oid,
        tokenUri,
        owner.address,
        i, // price increments with i
        1234567890, // same capExpires
        `${i * 10}`, // capDownloads
        `${i * 100}`, // capVolume
        `{key1: ${i * 10}}`, // cdsTarget
        `{key2: ${i * 100}}`, // cdsSl
        '0x00',
      );

      assets.push({ assetId, oid, tokenUri });
    }
    return assets;
  }

  beforeEach(async function () {
    [owner, addr1] = await ethers.getSigners();
    offeringToken = (await deployContractsForTests()).offeringTokenContract;
  });

  describe('addAsset and related functions tests', function () {
    it('should add multiple assets correctly', async function () {
      const testData = [
        {
          assetid: 'aabc1',
          oid: 'oabc1',
          resource: 'http://resource1.jpg',
          beneficiary: '0x5aCdD45C3edD6d70E2cb95676F5deDb71759BB13',
          price: 1,
          capExpires: 1234567890,
          capDownloads: '10',
          capVolume: '100',
          cdsTarget: '{key1: 10}',
          cdsSl: '{key2: 100}',
          tokenUri: 'http://tokenuri1.jpg',
        },
        {
          assetid: 'aabc2',
          oid: 'oabc2',
          resource: 'http://resource2.jpg',
          beneficiary: '0xe9BFD7BB36D2Ee90c695ACa030EF0bDB675c44E4',
          price: 2,
          capExpires: 1234567890,
          capDownloads: '20',
          capVolume: '200',
          cdsTarget: '{key1: 20}',
          cdsSl: '{key2: 200}',
          tokenUri: 'http://tokenuri2.jpg',
        },
      ];

      for (const data of testData) {
        const tx = await offeringToken.addAsset(
          data.assetid,
          data.oid,
          data.tokenUri,
          data.beneficiary,
          data.price,
          data.capExpires,
          data.capDownloads,
          data.capVolume,
          data.cdsTarget,
          data.cdsSl,
          '0x00',
        );

        await expect(tx).to.emit(offeringToken, 'TokenOfferingMinted');

        const assetidHash = await offeringToken.getIDHash(data.assetid);

        const exists = await offeringToken.tokenExists(data.oid);
        expect(exists).to.be.true;

        const dataProvider = await offeringToken.getOfferingDataProvider(
          data.oid,
        );
        expect(dataProvider).to.equal(data.beneficiary);

        const accessPrice = await offeringToken.getOfferDataAccessPrice(
          data.oid,
        );
        expect(accessPrice).to.equal(data.price);
      }
      // Check the total minted tokens after each addition
      const totalMinted = await offeringToken.getTotalMinted();
      expect(totalMinted).to.equal(2); // Should match the iteration count
    });

    it('should allow different beneficiary addresses for the same asset ID', async function () {
      const assetId = 'testAsset123';
      const beneficiary1 = '0xF2D72bF781C11186553884574C01175C7d205822';
      const beneficiary2 = '0x0347A5e25BBD6b7F235D21Afe27bb54287740045';

      // Add first offering with beneficiary1
      await offeringToken.addAsset(
        assetId,
        'offering1',
        'http://tokenuri1.jpg',
        beneficiary1,
        100,
        1234567890,
        '10',
        '100',
        '{content-id: "test1"}',
        '{metadata: "test1"}',
        '0x00',
      );

      // Add second offering for same asset with different beneficiary
      await offeringToken.addAsset(
        assetId,
        'offering2',
        'http://tokenuri2.jpg',
        beneficiary2,
        200,
        1234567890,
        '20',
        '200',
        '{content-id: "test2"}',
        '{metadata: "test2"}',
        '0x00',
      );

      // Verify both offerings exist
      const exists1 = await offeringToken.tokenExists('offering1');
      const exists2 = await offeringToken.tokenExists('offering2');
      expect(exists1).to.be.true;
      expect(exists2).to.be.true;

      // Verify different data providers for each offering
      const dataProvider1 =
        await offeringToken.getOfferingDataProvider('offering1');
      const dataProvider2 =
        await offeringToken.getOfferingDataProvider('offering2');
      expect(dataProvider1).to.equal(beneficiary1);
      expect(dataProvider2).to.equal(beneficiary2);

      // Verify the asset data provider is set to the latest one
      const assetDataProvider =
        await offeringToken.getAssetDataProvider(assetId);
      expect(assetDataProvider).to.equal(beneficiary2);

      // Verify all data providers array contains both providers
      const allAssetDataProviders =
        await offeringToken.getAllAssetDataProviders(assetId);
      expect(allAssetDataProviders).to.deep.equal([beneficiary1, beneficiary2]);
    });
    it('should return true for a token that historically exists and has not been burned', async function () {
      // Add an asset
      const assetId = 'aabc1';
      const oid = 'oabc1';
      await offeringToken.addAsset(
        assetId,
        oid,
        'http://tokenuri1.jpg',
        owner.address,
        1,
        1234567890,
        '10',
        '100',
        '{key1: 10}',
        '{key2: 100}',
        '0x00',
      );

      // Check if the token exists historically
      const exists = await offeringToken.historicallyTokenExists(oid);
      expect(exists).to.be.true;
    });

    it('should return true for historical existence for a token that has been burned', async function () {
      // Add an asset
      const assetId = 'aabc2';
      const oid = 'oabc2';
      await offeringToken.addAsset(
        assetId,
        oid,
        'http://tokenuri2.jpg',
        owner.address,
        2,
        1234567890,
        '20',
        '200',
        '{key1: 20}',
        '{key2: 200}',
        '0x00',
      );

      // Burn the token
      await offeringToken.removeOffering(oid);

      // Check if the token exists historically
      const exists = await offeringToken.historicallyTokenExists(oid);
      expect(exists).to.be.true;
    });

    it('should return false historical existence for a token that has never existed', async function () {
      const nonExistentOid = 'oabcNonExistent';

      // Check if the token exists historically
      const exists =
        await offeringToken.historicallyTokenExists(nonExistentOid);
      expect(exists).to.be.false;
    });

    it('should perform "get historically minted at" enumeration successfully', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Test `getHistoricallyMintedTokenAt`
      const bigNumbers1: BigNumber[] = [];
      const bigNumbers2: BigNumber[] = [];
      for (let i = 0; i < assetCount; i++) {
        const tokenIndx = await offeringToken.getHistoricallyMintedTokenAt(i);
        bigNumbers1.push(tokenIndx);
      }
      bigNumbers1.sort();
      for (let i = 0; i < assetCount; i++) {
        const oidHash = await offeringToken.getIDHash(assets[i].oid);
        bigNumbers2.push(oidHash);
      }
      bigNumbers2.sort();
      for (let i = 0; i < assetCount; i++) {
        expect(bigNumbers1[i]).to.equal(bigNumbers2[i]);
      }
    });

    it('should get offering token data provider address successfully', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Test `getOfferingTokenIndxDataProvider`
      for (const { oid } of assets) {
        const oidHash = await offeringToken.getIDHash(oid);
        const dataProvider =
          await offeringToken.getOfferingTokenIndxDataProvider(oidHash);
        expect(dataProvider).to.equal(owner.address);
      }
    });

    it('should check historically token existence for multiple token indices successfully', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Test `historicallyTokenExists` for existing tokens
      for (const { oid } of assets) {
        const exists = await offeringToken.historicallyTokenExists(oid);
        expect(exists).to.be.true;
      }
    });

    it('should get asset data provider successfully', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Test `getAssetDataProvider`
      for (const { assetId } of assets) {
        const assetHash = await offeringToken.getIDHash(assetId);
        const dataProvider = await offeringToken.getAssetDataProvider(assetId);
        expect(dataProvider).to.equal(owner.address);
      }
    });

    it('should get total number of historically minted tokens', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Test `getTotalMinted`
      const totalMinted = await offeringToken.getTotalMinted();
      expect(totalMinted).to.equal(assetCount);
    });

    it('should get number of minted and currently existing tokens', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Test `getMintedTokens`
      const mintedTokens = await offeringToken.getMintedTokens();
      expect(mintedTokens.length).to.equal(assetCount);
    });

    it('should get asset offering ids and their lengths successfully', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Test `getAssetOids`, `getNoOfAssetOids`, `getAssetOidAtIndex`
      for (const { assetId, oid } of assets) {
        const assetHash = await offeringToken.getIDHash(assetId);
        const oidHash = await offeringToken.getIDHash(oid);

        const assetOids = await offeringToken.getAssetOids(assetHash);
        expect(assetOids.length).to.equal(1);
        expect(assetOids[0]).to.equal(oidHash);

        const noOfAssetOids = await offeringToken.getNoOfAssetOids(assetHash);
        expect(noOfAssetOids).to.equal(1);

        const assetOidAtIndex = await offeringToken.getAssetOidAtIndex(
          assetHash,
          0,
        );
        expect(assetOidAtIndex).to.equal(oidHash);
      }
    });

    it('should perform token removal, and afterwards check current/historical token existence and get number of minted/burned tokens', async function () {
      const assetCount = 3; // Number of assets to test
      const assets = await addNMultipleAssets(assetCount);

      // Remove a token and test `removeOffering`, `getTotalBurned`, and `historicallyTokenExists`
      const { oid: removedOid } = assets[0];
      await offeringToken.removeOffering(removedOid);

      const totalBurned = await offeringToken.getTotalBurned();
      expect(totalBurned).to.equal(1);

      const stillExists =
        await offeringToken.historicallyTokenExists(removedOid);
      expect(stillExists).to.be.true;

      const currentlyExists = await offeringToken.tokenExists(removedOid);
      expect(currentlyExists).to.be.false;

      // Test `getHistoricallyMintedTokens`
      const historicalTokens =
        await offeringToken.getHistoricallyMintedTokens();
      expect(historicalTokens.length).to.equal(assetCount);

      // Ensure the removed token is not in the `getMintedTokens` list
      const updatedMintedTokens = await offeringToken.getMintedTokens();
      expect(updatedMintedTokens.length).to.equal(assetCount - 1);
    });
  });
});
